# LOTL Hunter - Environment Test Script
# Use this to validate your setup before deploying to production

Write-Host "LOTL Hunter Environment Test" -ForegroundColor Cyan
Write-Host "============================="

# Test 1: PowerShell Version
Write-Host "`n[1/5] Checking PowerShell version..." -ForegroundColor Yellow
$psVersion = $PSVersionTable.PSVersion
if ($psVersion.Major -ge 5) {
    Write-Host "   ✓ PowerShell $psVersion (minimum 5.1 required)" -ForegroundColor Green
} else {
    Write-Host "   ✗ PowerShell $psVersion - Upgrade to 5.1 or later" -ForegroundColor Red
}

# Test 2: Execution Policy
Write-Host "`n[2/5] Checking execution policy..." -ForegroundColor Yellow
$execPolicy = Get-ExecutionPolicy
$allowedPolicies = @("RemoteSigned", "Unrestricted", "Bypass")
if ($execPolicy -in $allowedPolicies) {
    Write-Host "   ✓ Execution policy: $execPolicy" -ForegroundColor Green
} else {
    Write-Host "   ⚠ Execution policy: $execPolicy (may block scripts)" -ForegroundColor Yellow
    Write-Host "   Consider: Set-ExecutionPolicy RemoteSigned -Scope CurrentUser" -ForegroundColor Gray
}

# Test 3: Administrative Privileges
Write-Host "`n[3/5] Checking privileges..." -ForegroundColor Yellow
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")
if ($isAdmin) {
    Write-Host "   ✓ Running as Administrator" -ForegroundColor Green
} else {
    Write-Host "   ⚠ Not running as Administrator (some checks may be limited)" -ForegroundColor Yellow
}

# Test 4: Required Modules
Write-Host "`n[4/5] Checking required modules..." -ForegroundColor Yellow
$modules = @("Microsoft.PowerShell.Security", "Microsoft.PowerShell.Management", "Microsoft.PowerShell.Utility")
foreach ($module in $modules) {
    if (Get-Module -ListAvailable -Name $module) {
        Write-Host "   ✓ $module available" -ForegroundColor Green
    } else {
        Write-Host "   ⚠ $module not found" -ForegroundColor Yellow
    }
}

# Test 5: Disk Space
Write-Host "`n[5/5] Checking disk space..." -ForegroundColor Yellow
$drive = Get-PSDrive -Name C -ErrorAction SilentlyContinue
if ($drive) {
    $freeGB = [math]::Round($drive.Free / 1GB, 1)
    if ($freeGB -gt 1) {
        Write-Host "   ✓ $freeGB GB free on C:" -ForegroundColor Green
    } else {
        Write-Host "   ⚠ Only $freeGB GB free on C: (recommended: 1GB+)" -ForegroundColor Yellow
    }
}

# Test 6: Sample Detection (Safe)
Write-Host "`n[6/6] Running safe sample detection..." -ForegroundColor Yellow
try {
    # Create a harmless test file
    $testFile = "$env:TEMP\lotl-test-$(Get-Random).txt"
    "This is a test file created by LOTL Hunter environment test." | Out-File -FilePath $testFile
    
    # Check if we can read it back
    $content = Get-Content -Path $testFile -ErrorAction SilentlyContinue
    if ($content) {
        Write-Host "   ✓ File system access working" -ForegroundColor Green
    }
    
    # Cleanup
    Remove-Item -Path $testFile -ErrorAction SilentlyContinue
    
} catch {
    Write-Host "   ⚠ File system test failed: $_" -ForegroundColor Yellow
}

# Summary
Write-Host "`n📋 Test Summary:" -ForegroundColor Cyan
Write-Host "   PowerShell $psVersion" -ForegroundColor $(if ($psVersion.Major -ge 5) { "Green" } else { "Red" })
Write-Host "   Execution policy: $execPolicy" -ForegroundColor $(if ($execPolicy -in $allowedPolicies) { "Green" } else { "Yellow" })
Write-Host "   Admin privileges: $(if ($isAdmin) { 'Yes' } else { 'No' })" -ForegroundColor $(if ($isAdmin) { "Green" } else { "Yellow" })
Write-Host "   Disk space: $freeGB GB free" -ForegroundColor $(if ($freeGB -gt 1) { "Green" } else { "Yellow" })

if ($psVersion.Major -ge 5 -and $execPolicy -in $allowedPolicies) {
    Write-Host "`n✅ Environment ready for LOTL Hunter deployment" -ForegroundColor Green
    Write-Host "`nNext steps:"
    Write-Host "1. Run .\Quick-Start.ps1 from Detection_Engine folder" -ForegroundColor Cyan
    Write-Host "2. Review the generated report" -ForegroundColor Cyan
    Write-Host "3. Schedule daily scans via Task Scheduler" -ForegroundColor Cyan
} else {
    Write-Host "`n⚠ Environment needs adjustment before deployment" -ForegroundColor Yellow
    Write-Host "`nRecommended fixes:" -ForegroundColor Cyan
    if ($psVersion.Major -lt 5) {
        Write-Host "- Upgrade PowerShell to version 5.1 or later" -ForegroundColor Gray
    }
    if ($execPolicy -notin $allowedPolicies) {
        Write-Host "- Set execution policy: Set-ExecutionPolicy RemoteSigned -Scope CurrentUser" -ForegroundColor Gray
    }
    if (-not $isAdmin) {
        Write-Host "- Run PowerShell as Administrator for full functionality" -ForegroundColor Gray
    }
}

Write-Host "`n💡 Tip: Test on a non-production system first!" -ForegroundColor Gray
Write-Host "`nCyborama, LLC - 'Threat Intelligence with Teeth™'" -ForegroundColor DarkCyan