# LOT-Squatch Quick Start
# Run this to perform a basic LOTL scan

Write-Host "LOTL Hunter - Quick Start" -ForegroundColor Cyan
Write-Host "========================="

# Check if running as administrator
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")
if (-not $isAdmin) {
    Write-Host "WARNING: Run as Administrator for best results" -ForegroundColor Yellow
}

# Import main detection module
try {
    . ".\LOT-Squatch.ps1"
    Write-Host "✓ LOT-Squatch loaded" -ForegroundColor Green
} catch {
    Write-Host "✗ Failed to load LOT-Squatch: $_" -ForegroundColor Red
    exit 1
}

# Run basic scan
Write-Host "Running LOTL detection..." -ForegroundColor Yellow
Write-Host "This may take 1-2 minutes..." -ForegroundColor Gray

$results = Invoke-LOTLDetection -QuickScan

# Display results summary
$highCount = ($results | Where-Object { $_.Risk -eq "High" }).Count
$mediumCount = ($results | Where-Object { $_.Risk -eq "Medium" }).Count
$lowCount = ($results | Where-Object { $_.Risk -eq "Low" }).Count

Write-Host "`n📊 Scan Results:" -ForegroundColor Cyan
Write-Host "   High risk:   $highCount" -ForegroundColor $(if ($highCount -gt 0) { "Red" } else { "Green" })
Write-Host "   Medium risk: $mediumCount" -ForegroundColor $(if ($mediumCount -gt 0) { "Yellow" } else { "Green" })
Write-Host "   Low risk:    $lowCount" -ForegroundColor $(if ($lowCount -gt 0) { "Green" } else { "Gray" })
Write-Host "   Total:       $($results.Count)" -ForegroundColor Cyan

if ($results.Count -gt 0) {
    Write-Host "`n🔍 Top findings:" -ForegroundColor Yellow
    $results | Select-Object -First 3 | Format-Table Title, Risk, Category -AutoSize
    
    # Generate report
    $reportPath = ".\LOTL-Report-$(Get-Date -Format 'yyyy-MM-dd').html"
    $results | ConvertTo-Html -Title "LOTL Scan Report" | Out-File -FilePath $reportPath
    Write-Host "`n📄 Report saved: $reportPath" -ForegroundColor Green
    
    Write-Host "`n🚨 Next steps:" -ForegroundColor Yellow
    Write-Host "1. Review the HTML report for details"
    Write-Host "2. Investigate high-risk findings first"
    Write-Host "3. Check false positives in the guide"
    Write-Host "4. Schedule daily scans via Task Scheduler"
} else {
    Write-Host "`n✅ No LOTL findings detected" -ForegroundColor Green
    Write-Host "`n📅 Next steps:" -ForegroundColor Cyan
    Write-Host "1. Schedule daily scans via Task Scheduler"
    Write-Host "2. Set up the Flask dashboard for monitoring"
    Write-Host "3. Test with sample data (see Samples folder)"
}

Write-Host "`n💡 Tip: Run with -FullScan for comprehensive detection" -ForegroundColor Gray
Write-Host "`nCyborama, LLC - 'Threat Intelligence with Teeth™'" -ForegroundColor DarkCyan