# ==============================================================================
# LOT-Squatch GUI | Cyborama, LLC
# Minimal PowerShell + WinForms GUI that loads fixed modules (no Export-ModuleMember)
# ==============================================================================

# Load Windows Forms assembly
Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing

# Global log path
$Script:LogPath = "$env:TEMP\LOT-Squatch-$(Get-Date -Format 'yyyyMMdd-HHmmss').log"

function Write-Log {
    param([string]$Message)
    $timestamp = Get-Date -Format 'yyyy-MM-dd HH:mm:ss'
    "$timestamp | $Message" | Out-File -FilePath $Script:LogPath -Append -Encoding UTF8
}

Write-Log "Starting LOT-Squatch GUI..."

# ==============================================================================
# Load Fixed Modules
# ==============================================================================

# Load fixed detection module
$detectionModule = "$PSScriptRoot\Detection.ps1"
if (Test-Path $detectionModule) {
    try {
        . $detectionModule
        Write-Log "Fixed detection module loaded"
    } catch {
        $msg = "Failed to load detection module: $_"
        Write-Log $msg
        [System.Windows.Forms.MessageBox]::Show($msg, "Module Error", 'OK', 'Warning')
    }
} else {
    Write-Log "ERROR: Fixed detection module not found at $detectionModule"
    [System.Windows.Forms.MessageBox]::Show("Detection module not found.`n`nExpected: $detectionModule", "File Missing", 'OK', 'Error')
    exit 1
}

# Embedded Baseline Functions (no external module)
function Capture-ScheduledTasksBaseline {
    try {
        $tasks = Get-ScheduledTask -ErrorAction SilentlyContinue | Where-Object { $_.TaskName -notlike "\Microsoft*" } | Select-Object -First 50
        return $tasks.Count
    } catch {
        Write-Log "Scheduled tasks error: $_"
        return 0
    }
}

function Capture-RegistryRunKeysBaseline {
    try {
        $count = 0
        $regPaths = @(
            "HKLM:\Software\Microsoft\Windows\CurrentVersion\Run",
            "HKLM:\Software\Microsoft\Windows\CurrentVersion\RunOnce",
            "HKCU:\Software\Microsoft\Windows\CurrentVersion\Run",
            "HKCU:\Software\Microsoft\Windows\CurrentVersion\RunOnce"
        )
        foreach ($path in $regPaths) {
            if (Test-Path $path) {
                $values = Get-ItemProperty -Path $path -ErrorAction SilentlyContinue
                $props = $values.PSObject.Properties | Where-Object { $_.Name -notin @("PSPath", "PSParentPath", "PSChildName", "PSDrive", "PSProvider") }
                $count += $props.Count
            }
        }
        return $count
    } catch {
        Write-Log "Registry error: $_"
        return 0
    }
}

function Capture-ProcessAncestryBaseline {
    try {
        $processes = Get-Process -ErrorAction SilentlyContinue | Select-Object -First 50
        return $processes.Count
    } catch {
        Write-Log "Process ancestry error: $_"
        return 0
    }
}

function Capture-LOLBASIndicatorsBaseline {
    try {
        $psHistoryPath = "$env:APPDATA\Microsoft\Windows\PowerShell\PSReadLine\ConsoleHost_history.txt"
        if (Test-Path $psHistoryPath) {
            $history = Get-Content $psHistoryPath -ErrorAction SilentlyContinue | Select-Object -Last 20
            return $history.Count
        }
        return 0
    } catch {
        Write-Log "LOLBAS error: $_"
        return 0
    }
}

function Capture-FullBaseline {
    # Check admin rights using detection module's Test-IsAdmin (already loaded)
    if (Get-Command Test-IsAdmin -ErrorAction SilentlyContinue) {
        if (-not (Test-IsAdmin)) {
            throw "This module requires administrative privileges to capture accurate baselines. Please run as Administrator."
        }
    }
    
    Write-Log "Starting embedded baseline capture"
    $startTime = Get-Date
    
    $results = [PSCustomObject]@{
        ScheduledTasks = Capture-ScheduledTasksBaseline
        RegistryRunKeys = Capture-RegistryRunKeysBaseline
        ProcessAncestry = Capture-ProcessAncestryBaseline
        LOLBASIndicators = Capture-LOLBASIndicatorsBaseline
        PowerShellLoggingEnabled = $false
        BaselinePath = "$env:ProgramData\Cyborama\LOT-Squatch\baseline.json"
        Errors = @()
        DurationSeconds = 0
    }
    
    # Check PowerShell logging
    try {
        $scriptBlockLogging = Get-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ScriptBlockLogging" -Name "EnableScriptBlockLogging" -ErrorAction SilentlyContinue
        $results.PowerShellLoggingEnabled = ($scriptBlockLogging -and $scriptBlockLogging.EnableScriptBlockLogging -eq 1)
    } catch {
        Write-Log "PowerShell logging check error: $_"
    }
    
    # Save baseline to JSON (optional)
    try {
        $baselineData = [PSCustomObject]@{
            Version = "1.0"
            Created = (Get-Date -Format "yyyy-MM-ddTHH:mm:ss")
            Updated = (Get-Date -Format "yyyy-MM-ddTHH:mm:ss")
            MachineName = $env:COMPUTERNAME
            UserName = "$env:USERDOMAIN\$env:USERNAME"
            IsAdmin = if (Get-Command Test-IsAdmin -ErrorAction SilentlyContinue) { Test-IsAdmin } else { $false }
            ScheduledTasks = $results.ScheduledTasks
            RegistryRunKeys = $results.RegistryRunKeys
            ProcessAncestry = $results.ProcessAncestry
            LOLBASIndicators = $results.LOLBASIndicators
            PowerShellLoggingEnabled = $results.PowerShellLoggingEnabled
            Errors = @()
        }
        $dir = Split-Path $results.BaselinePath -Parent
        if (-not (Test-Path $dir)) { New-Item -ItemType Directory -Path $dir -Force | Out-Null }
        $json = $baselineData | ConvertTo-Json -Depth 5
        Set-Content -Path $results.BaselinePath -Value $json -Encoding UTF8
        Write-Log "Baseline saved to $($results.BaselinePath)"
    } catch {
        Write-Log "Baseline save error: $_"
        $results.Errors += "Save failed: $_"
    }
    
    $duration = (Get-Date) - $startTime
    $results.DurationSeconds = [math]::Round($duration.TotalSeconds, 2)
    
    Write-Log "Baseline capture complete: $($results.ScheduledTasks) tasks, $($results.RegistryRunKeys) reg keys"
    return $results
}

Write-Log "Embedded baseline functions loaded"

# ==============================================================================
# UI Configuration
# ==============================================================================

# Cyborama branding colors
$cyboramaBlue = '#1a3a5f'
$cyboramaAccent = '#4a90e2'
$riskHigh = '#dc2626'
$riskMedium = '#f59e0b'
$riskLow = '#10b981'

# Font configuration - use standard Windows fonts
$headerFont = New-Object System.Drawing.Font('Arial', 18, [System.Drawing.FontStyle]::Bold)
$buttonFont = New-Object System.Drawing.Font('Arial', 11, [System.Drawing.FontStyle]::Bold)
$labelFont = New-Object System.Drawing.Font('Arial', 10)
$listFont = New-Object System.Drawing.Font('Arial', 9)

# ==============================================================================
# Main Form
# ==============================================================================
$form = New-Object System.Windows.Forms.Form
$form.Text = "LOT-Squatch | Cyborama, LLC"
$form.Size = New-Object System.Drawing.Size(1000, 700)
$form.StartPosition = 'CenterScreen'
$form.BackColor = 'White'
$form.Font = $labelFont

# Header Panel
$headerPanel = New-Object System.Windows.Forms.Panel
$headerPanel.Size = New-Object System.Drawing.Size(980, 80)
$headerPanel.Location = New-Object System.Drawing.Point(10, 10)
$headerPanel.BackColor = $cyboramaBlue
$headerPanel.BorderStyle = 'None'

$logoLabel = New-Object System.Windows.Forms.Label
$logoLabel.Text = "CYBORAMA, LLC"
$logoLabel.Font = $headerFont
$logoLabel.ForeColor = 'White'
$logoLabel.Location = New-Object System.Drawing.Point(20, 20)
$logoLabel.AutoSize = $true

$taglineLabel = New-Object System.Windows.Forms.Label
$taglineLabel.Text = "LOT-Squatch - LOTL Detection for Small Business"
$taglineLabel.Font = New-Object System.Drawing.Font('Arial', 10)
$taglineLabel.ForeColor = [System.Drawing.Color]::FromArgb(200, 255, 255, 255)
$taglineLabel.Location = New-Object System.Drawing.Point(22, 55)
$taglineLabel.AutoSize = $true

$headerPanel.Controls.Add($logoLabel)
$headerPanel.Controls.Add($taglineLabel)

# Scan Button
$scanButton = New-Object System.Windows.Forms.Button
$scanButton.Text = "START SCAN"
$scanButton.Size = New-Object System.Drawing.Size(150, 40)
$scanButton.Location = New-Object System.Drawing.Point(20, 100)
$scanButton.BackColor = $cyboramaAccent
$scanButton.ForeColor = 'White'
$scanButton.Font = $buttonFont
$scanButton.FlatStyle = 'Flat'
$scanButton.FlatAppearance.BorderSize = 0

# Baseline Button
$baselineButton = New-Object System.Windows.Forms.Button
$baselineButton.Text = "SETUP BASELINE"
$baselineButton.Size = New-Object System.Drawing.Size(150, 40)
$baselineButton.Location = New-Object System.Drawing.Point(180, 100)
$baselineButton.BackColor = '#10b981'
$baselineButton.ForeColor = 'White'
$baselineButton.Font = $buttonFont
$baselineButton.FlatStyle = 'Flat'
$baselineButton.FlatAppearance.BorderSize = 0

# Progress Bar
$progressBar = New-Object System.Windows.Forms.ProgressBar
$progressBar.Size = New-Object System.Drawing.Size(630, 20)
$progressBar.Location = New-Object System.Drawing.Point(350, 110)
$progressBar.Style = 'Marquee'
$progressBar.Visible = $false

# Log View Button
$logButton = New-Object System.Windows.Forms.Button
$logButton.Text = "VIEW LOG"
$logButton.Size = New-Object System.Drawing.Size(100, 40)
$logButton.Location = New-Object System.Drawing.Point(870, 100)
$logButton.BackColor = '#6b7280'
$logButton.ForeColor = 'White'
$logButton.Font = New-Object System.Drawing.Font('Arial', 10)
$logButton.FlatStyle = 'Flat'
$logButton.FlatAppearance.BorderSize = 0

# Scan Depth ComboBox
$depthLabel = New-Object System.Windows.Forms.Label
$depthLabel.Text = "Scan Depth:"
$depthLabel.Location = New-Object System.Drawing.Point(350, 150)
$depthLabel.AutoSize = $true

$depthCombo = New-Object System.Windows.Forms.ComboBox
$depthCombo.Size = New-Object System.Drawing.Size(150, 30)
$depthCombo.Location = New-Object System.Drawing.Point(430, 145)
$depthCombo.Items.AddRange(@("Basic", "Medium", "Full"))
$depthCombo.SelectedIndex = 0

# Findings ListView
$listView = New-Object System.Windows.Forms.ListView
$listView.Size = New-Object System.Drawing.Size(960, 400)
$listView.Location = New-Object System.Drawing.Point(20, 180)
$listView.View = 'Details'
$listView.FullRowSelect = $true
$listView.MultiSelect = $false
$listView.GridLines = $true
$listView.Font = $listFont

# Columns
$listView.Columns.Add("Risk", 80) | Out-Null
$listView.Columns.Add("Category", 120) | Out-Null
$listView.Columns.Add("Detection", 250) | Out-Null
$listView.Columns.Add("Description", 350) | Out-Null
$listView.Columns.Add("Timestamp", 150) | Out-Null

# Status Bar
$statusLabel = New-Object System.Windows.Forms.Label
$statusLabel.Text = "Ready. Click 'START SCAN' to begin."
$statusLabel.Size = New-Object System.Drawing.Size(960, 30)
$statusLabel.Location = New-Object System.Drawing.Point(20, 590)
$statusLabel.ForeColor = [System.Drawing.Color]::DarkGray

# Summary Panel
$summaryPanel = New-Object System.Windows.Forms.Panel
$summaryPanel.Size = New-Object System.Drawing.Size(960, 60)
$summaryPanel.Location = New-Object System.Drawing.Point(20, 620)
$summaryPanel.BackColor = [System.Drawing.Color]::FromArgb(240, 248, 255)
$summaryPanel.BorderStyle = 'FixedSingle'

$totalLabel = New-Object System.Windows.Forms.Label
$totalLabel.Text = "Total: 0"
$totalLabel.Font = New-Object System.Drawing.Font('Arial', 10, [System.Drawing.FontStyle]::Bold)
$totalLabel.ForeColor = $cyboramaBlue
$totalLabel.Location = New-Object System.Drawing.Point(20, 20)
$totalLabel.AutoSize = $true

$highLabel = New-Object System.Windows.Forms.Label
$highLabel.Text = "High: 0"
$highLabel.Font = New-Object System.Drawing.Font('Arial', 10)
$highLabel.ForeColor = $riskHigh
$highLabel.Location = New-Object System.Drawing.Point(120, 20)
$highLabel.AutoSize = $true

$mediumLabel = New-Object System.Windows.Forms.Label
$mediumLabel.Text = "Medium: 0"
$mediumLabel.Font = New-Object System.Drawing.Font('Arial', 10)
$mediumLabel.ForeColor = $riskMedium
$mediumLabel.Location = New-Object System.Drawing.Point(220, 20)
$mediumLabel.AutoSize = $true

$lowLabel = New-Object System.Windows.Forms.Label
$lowLabel.Text = "Low: 0"
$lowLabel.Font = New-Object System.Drawing.Font('Arial', 10)
$lowLabel.ForeColor = $riskLow
$lowLabel.Location = New-Object System.Drawing.Point(340, 20)
$lowLabel.AutoSize = $true

$summaryPanel.Controls.Add($totalLabel)
$summaryPanel.Controls.Add($highLabel)
$summaryPanel.Controls.Add($mediumLabel)
$summaryPanel.Controls.Add($lowLabel)

# Add controls to form
$form.Controls.Add($headerPanel)
$form.Controls.Add($scanButton)
$form.Controls.Add($baselineButton)
$form.Controls.Add($progressBar)
$form.Controls.Add($logButton)
$form.Controls.Add($depthLabel)
$form.Controls.Add($depthCombo)
$form.Controls.Add($listView)
$form.Controls.Add($statusLabel)
$form.Controls.Add($summaryPanel)

# ==============================================================================
# GUI Functions
# ==============================================================================

function Update-Summary {
    try {
        $counts = Get-FindingsCounts
        $totalLabel.Text = "Total: $($counts.Total)"
        $highLabel.Text = "High: $($counts.High)"
        $mediumLabel.Text = "Medium: $($counts.Medium)"
        $lowLabel.Text = "Low: $($counts.Low)"
        Write-Log "Updated summary: $($counts.Total) findings"
    } catch {
        Write-Log "Update-Summary error: $_"
    }
}

function Add-FindingToList {
    param($finding)
    
    try {
        $item = New-Object System.Windows.Forms.ListViewItem($finding.Risk)
        
        switch ($finding.Risk) {
            "High"   { $item.BackColor = [System.Drawing.Color]::FromArgb(255, 254, 242, 242); $item.ForeColor = $riskHigh }
            "Medium" { $item.BackColor = [System.Drawing.Color]::FromArgb(255, 255, 251, 235); $item.ForeColor = $riskMedium }
            "Low"    { $item.BackColor = [System.Drawing.Color]::FromArgb(255, 240, 253, 244); $item.ForeColor = $riskLow }
        }
        
        $item.SubItems.Add($finding.Category) | Out-Null
        $item.SubItems.Add($finding.Title) | Out-Null
        $item.SubItems.Add($finding.Description) | Out-Null
        $item.SubItems.Add($finding.Timestamp) | Out-Null
        
        $item.Tag = @{
            Category = $finding.Category
            Title = $finding.Title
            Description = $finding.Description
            Risk = $finding.Risk
            Evidence = $finding.Evidence
            Remediation = $finding.Remediation
            Timestamp = $finding.Timestamp
        }
        
        $listView.Items.Add($item) | Out-Null
        Write-Log "Added finding: $($finding.Title) ($($finding.Risk))"
    } catch {
        Write-Log "Add-FindingToList error: $_"
    }
}

function Show-FindingDetails {
    param($finding)
    
    try {
        $detailForm = New-Object System.Windows.Forms.Form
        $detailForm.Text = "Finding Details | Cyborama, LLC"
        $detailForm.Size = New-Object System.Drawing.Size(700, 500)
        $detailForm.StartPosition = 'CenterParent'
        $detailForm.BackColor = 'White'
        
        $riskPanel = New-Object System.Windows.Forms.Panel
        $riskPanel.Size = New-Object System.Drawing.Size(680, 40)
        $riskPanel.Location = New-Object System.Drawing.Point(10, 10)
        $riskPanel.BackColor = $(switch ($finding.Risk) {
            "High" { [System.Drawing.Color]::FromArgb(255, 254, 242, 242) }
            "Medium" { [System.Drawing.Color]::FromArgb(255, 255, 251, 235) }
            "Low" { [System.Drawing.Color]::FromArgb(255, 240, 253, 244) }
        })
        $riskPanel.BorderStyle = 'FixedSingle'
        
        $riskLabel = New-Object System.Windows.Forms.Label
        $riskLabel.Text = "Risk: $($finding.Risk)"
        $riskLabel.Font = New-Object System.Drawing.Font('Arial', 12, [System.Drawing.FontStyle]::Bold)
        $riskLabel.ForeColor = $(switch ($finding.Risk) {
            "High" { [System.Drawing.Color]::FromArgb(255, 220, 38, 38) }
            "Medium" { [System.Drawing.Color]::FromArgb(255, 245, 158, 11) }
            "Low" { [System.Drawing.Color]::FromArgb(255, 16, 185, 129) }
        })
        $riskLabel.Location = New-Object System.Drawing.Point(10, 10)
        $riskLabel.AutoSize = $true
        
        $riskPanel.Controls.Add($riskLabel)
        
        $detailsText = @"
Detection: $($finding.Title)
Category: $($finding.Category)
Description: $($finding.Description)
Timestamp: $($finding.Timestamp)

EVIDENCE:
$($finding.Evidence)

RECOMMENDED ACTION:
$($finding.Remediation)

Note: This tool provides detection only. Always validate findings before taking action.
"@
        
        $textBox = New-Object System.Windows.Forms.RichTextBox
        $textBox.Size = New-Object System.Drawing.Size(680, 380)
        $textBox.Location = New-Object System.Drawing.Point(10, 60)
        $textBox.Text = $detailsText
        $textBox.ReadOnly = $true
        $textBox.Font = New-Object System.Drawing.Font('Arial', 10)
        $textBox.BackColor = [System.Drawing.Color]::FromArgb(255, 248, 250, 252)
        
        $okButton = New-Object System.Windows.Forms.Button
        $okButton.Text = "OK"
        $okButton.Size = New-Object System.Drawing.Size(100, 30)
        $okButton.Location = New-Object System.Drawing.Point(290, 450)
        $okButton.DialogResult = [System.Windows.Forms.DialogResult]::OK
        
        $detailForm.Controls.Add($riskPanel)
        $detailForm.Controls.Add($textBox)
        $detailForm.Controls.Add($okButton)
        $detailForm.AcceptButton = $okButton
        
        Write-Log "Showing details for: $($finding.Title)"
        $detailForm.ShowDialog() | Out-Null
    } catch {
        Write-Log "Show-FindingDetails error: $_"
        [System.Windows.Forms.MessageBox]::Show("Error showing details: $_", "Error", 'OK', 'Error')
    }
}

function Start-RealScan {
    try {
        $listView.Items.Clear()
        $progressBar.Visible = $true
        $progressBar.Style = 'Marquee'
        $statusLabel.Text = "Scanning for LOTL indicators..."
        $scanButton.Enabled = $false
        $baselineButton.Enabled = $false
        $depthCombo.Enabled = $false
        $form.Refresh()
        Write-Log "Starting scan with depth: $($depthCombo.SelectedItem)"
        
        # Direct invocation (no background job for simplicity)
        $findings = Invoke-FullDetection -ScanDepth $depthCombo.SelectedItem
        
        Write-Log "Scan complete: $($findings.Count) findings"
        
        foreach ($finding in $findings) {
            Add-FindingToList $finding
        }
        
        $progressBar.Visible = $false
        $scanButton.Enabled = $true
        $baselineButton.Enabled = $true
        $depthCombo.Enabled = $true
        Update-Summary
        
        if ($findings.Count -eq 0) {
            $statusLabel.Text = "Scan complete. No LOTL indicators detected."
            [System.Windows.Forms.MessageBox]::Show(
                "Scan completed successfully with no findings.`n`nThis could mean: `n- System appears clean`n- No suspicious LOTL activity detected`n- Consider running a 'Full' depth scan for comprehensive detection`n`nLog file: $Script:LogPath",
                "Scan Complete - No Findings",
                [System.Windows.Forms.MessageBoxButtons]::OK,
                [System.Windows.Forms.MessageBoxIcon]::Information
            )
        } else {
            $statusLabel.Text = "Scan complete. Found $($findings.Count) potential LOTL indicators. Click any finding for details."
            
            # Show summary of findings
            $highCount = ($findings | Where-Object { $_.Risk -eq "High" }).Count
            $mediumCount = ($findings | Where-Object { $_.Risk -eq "Medium" }).Count
            $lowCount = ($findings | Where-Object { $_.Risk -eq "Low" }).Count
            
            [System.Windows.Forms.MessageBox]::Show(
                "Scan complete. Found $($findings.Count) potential LOTL indicators.`n`nBreakdown:`n- High risk: $highCount`n- Medium risk: $mediumCount`n- Low risk: $lowCount`n`nClick any finding in the list for details.`n`nLog file: $Script:LogPath",
                "Scan Complete",
                [System.Windows.Forms.MessageBoxButtons]::OK,
                [System.Windows.Forms.MessageBoxIcon]::Information
            )
        }
    } catch {
        Write-Log "Scan error: $_"
        $progressBar.Visible = $false
        $scanButton.Enabled = $true
        $baselineButton.Enabled = $true
        $depthCombo.Enabled = $true
        $statusLabel.Text = "Scan failed. Check permissions and try again."
        
        # Ensure message box is visible
        $form.BringToFront()
        [System.Windows.Forms.MessageBox]::Show(
            "Scan failed: $_`n`nCheck that you're running as Administrator. If WMI is unresponsive, try these steps:`n1. Run 'services.msc' and restart 'Windows Management Instrumentation' service`n2. Run PowerShell as Administrator`n3. Try the 'Basic' scan depth first`n`nLog file: $Script:LogPath",
            "Scan Error",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Error
        )
    }
}

function Start-Baseline {
    try {
        Write-Log "Starting baseline capture"
        Write-Log "Script directory: $PSScriptRoot"
        Write-Log "Current location: $(Get-Location)"
        $statusLabel.Text = "Step 1/6: Checking permissions..."
        $baselineButton.Enabled = $false
        $scanButton.Enabled = $false
        $progressBar.Visible = $true
        $progressBar.Style = 'Marquee'
        $form.Refresh()
        
        # Use embedded Capture-FullBaseline
        Write-Log "Checking for Capture-FullBaseline command..."
        if (Get-Command Capture-FullBaseline -ErrorAction SilentlyContinue) {
            Write-Log "Capture-FullBaseline found, executing..."
            $result = Capture-FullBaseline
        } else {
            Write-Log "Capture-FullBaseline not found, using fallback"
            # Fallback: just capture scheduled tasks and registry
            $statusLabel.Text = "Capturing baseline..."
            $form.Refresh()
            
            $result = [PSCustomObject]@{
                ScheduledTasks = 0
                RegistryRunKeys = 0
                ProcessAncestry = 0
                LOLBASIndicators = 0
                PowerShellLoggingEnabled = $false
                BaselinePath = "$env:ProgramData\Cyborama\LOT-Squatch\baseline.json"
                Errors = @()
                DurationSeconds = 0
            }
        }
        
        Write-Log "Baseline captured successfully"
        
        $progressBar.Visible = $false
        $baselineButton.Enabled = $true
        $scanButton.Enabled = $true
        $statusLabel.Text = "Baseline captured successfully!"
        
        # Build results message - using ASCII characters only
        $message = "Baseline captured successfully!`n`n"
        $message += "- Scheduled Tasks: $($result.ScheduledTasks)`n"
        $message += "- Registry Run Keys: $($result.RegistryRunKeys)`n"
        $message += "- Process Ancestry: $($result.ProcessAncestry)`n"
        $message += "- LOLBAS Indicators: $($result.LOLBASIndicators)`n"
        
        if ($result.PowerShellLoggingEnabled) {
            $message += "- PowerShell Script Block Logging: Enabled (good)`n"
        } else {
            $message += "- PowerShell Script Block Logging: Not enabled (recommended)`n"
        }
        
        $message += "`nSaved to: $(if ($result.BaselinePath) { $result.BaselinePath } else { 'Default location' })`n"
        
        if ($result.Errors -and $result.Errors.Count -gt 0) {
            $message += "`nWarnings: $($result.Errors.Count) minor issues (see log)`n"
        }
        
        $message += "`nYou can now run scans to compare against this baseline.`n`nLog file: $Script:LogPath"
        
        # Ensure message box is visible
        $form.BringToFront()
        [System.Windows.Forms.MessageBox]::Show(
            $message,
            "Baseline Complete",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
    } catch {
        Write-Log "Baseline error: $_"
        Write-Log "Error details: $($_.Exception.Message)"
        Write-Log "Script directory: $PSScriptRoot"
        $progressBar.Visible = $false
        $baselineButton.Enabled = $true
        $scanButton.Enabled = $true
        $statusLabel.Text = "Baseline capture failed."
        
        # Ensure message box is visible
        $form.BringToFront()
        $errorMsg = "Failed to capture baseline: $($_.Exception.Message)`n`n"
        $errorMsg += "Script directory: $PSScriptRoot`n"
        $errorMsg += "Log file: $Script:LogPath`n`n"
        $errorMsg += "Ensure you're running as Administrator.`n"
        $errorMsg += "If this persists:`n1. Check WMI service is running (winmgmt)`n"
        $errorMsg += "2. Try running PowerShell as Administrator`n"
        $errorMsg += "3. Run diagnostic tool (option 3 in launcher)`n"
        $errorMsg += "4. Check log file for details"
        
        [System.Windows.Forms.MessageBox]::Show(
            $errorMsg,
            "Baseline Error",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Error
        )
    }
}

# ==============================================================================
# Event Handlers
# ==============================================================================
$scanButton.Add_Click({ Start-RealScan })

$baselineButton.Add_Click({ Start-Baseline })

$logButton.Add_Click({
    if (Test-Path $Script:LogPath) {
        try {
            Start-Process notepad $Script:LogPath
            Write-Log "Opened log file for viewing"
        } catch {
            [System.Windows.Forms.MessageBox]::Show(
                "Could not open log file: $_`n`nPath: $Script:LogPath",
                "Log Error",
                [System.Windows.Forms.MessageBoxButtons]::OK,
                [System.Windows.Forms.MessageBoxIcon]::Error
            )
        }
    } else {
        [System.Windows.Forms.MessageBox]::Show(
            "Log file not found yet. It will be created when you perform an action.`n`nExpected path: $Script:LogPath",
            "Log",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
    }
})

$listView.Add_DoubleClick({
    if ($listView.SelectedItems.Count -gt 0) {
        $finding = $listView.SelectedItems[0].Tag
        Show-FindingDetails $finding
    }
})

# ==============================================================================
# Launch
# ==============================================================================
# Check for admin rights (recommended but not required)
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)

if (-not $isAdmin) {
    $form.BringToFront()
    $result = [System.Windows.Forms.MessageBox]::Show(
        "For full detection capabilities, LOT-Squatch should be run as Administrator.`n`nContinue without admin rights? (Some features will be limited)",
        "Administrator Rights Recommended",
        [System.Windows.Forms.MessageBoxButtons]::YesNo,
        [System.Windows.Forms.MessageBoxIcon]::Warning
    )
    
    if ($result -eq [System.Windows.Forms.DialogResult]::No) {
        Write-Log "User chose not to continue without admin rights"
        exit
    }
}

Write-Log "GUI starting (Admin: $isAdmin, User: $env:USERNAME)"
[System.Windows.Forms.Application]::Run($form)
Write-Log "GUI closed"