# ==============================================================================
# LOT-Squatch Detection Module | Cyborama, LLC
# PowerShell functions for LOTL detection with reduced false positives
# ==============================================================================

# Global findings array
$Script:Findings = @()
$Script:HighCount = 0
$Script:MediumCount = 0
$Script:LowCount = 0

# Allowlist for known benign patterns
$Script:AllowlistPatterns = @(
    ".*WindowsPowerShell.*",
    ".*Microsoft.*",
    ".*Intel.*",
    ".*Dell.*",
    ".*HP.*",
    ".*Lenovo.*",
    ".*Adobe.*",
    ".*Java.*",
    ".*Google.*",
    ".*Chrome.*",
    ".*Firefox.*",
    ".*Mozilla.*",
    ".*NVIDIA.*",
    ".*AMD.*",
    ".*Kaspersky.*",
    ".*McAfee.*",
    ".*Symantec.*",
    ".*TrendMicro.*",
    ".*CarbonBlack.*",
    ".*CrowdStrike.*",
    ".*SentinelOne.*"
)

function Add-Finding {
    param(
        [string]$Category,
        [string]$Title,
        [string]$Description,
        [string]$Risk = "Medium",
        [string]$Evidence = "",
        [string]$Remediation = ""
    )
    
    $finding = [PSCustomObject]@{
        Category = $Category
        Title = $Title
        Description = $Description
        Risk = $Risk
        Evidence = $Evidence
        Remediation = $Remediation
        Timestamp = Get-Date -Format "HH:mm:ss"
    }
    
    $Script:Findings += $finding
    
    switch ($Risk) {
        "High"   { $Script:HighCount++ }
        "Medium" { $Script:MediumCount++ }
        "Low"    { $Script:LowCount++ }
    }
    
    return $finding
}

function Clear-Findings {
    $Script:Findings = @()
    $Script:HighCount = 0
    $Script:MediumCount = 0
    $Script:LowCount = 0
}

function Get-Findings {
    return $Script:Findings
}

function Get-FindingsCounts {
    return [PSCustomObject]@{
        Total = $Script:Findings.Count
        High = $Script:HighCount
        Medium = $Script:MediumCount
        Low = $Script:LowCount
    }
}

function Test-IsAdmin {
    return ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

function Test-IsAllowlisted {
    param([string]$InputString)
    
    foreach ($pattern in $Script:AllowlistPatterns) {
        if ($InputString -match $pattern) {
            return $true
        }
    }
    return $false
}

function Decode-Base64IfPossible {
    param([string]$EncodedString)
    
    try {
        # Try to extract Base64 from common patterns
        if ($EncodedString -match "powershell.*-EncodedCommand\s+['\""]?([A-Za-z0-9+/=]+)['\""]?") {
            $b64 = $matches[1]
        } elseif ($EncodedString -match "-e\s+['\""]?([A-Za-z0-9+/=]+)['\""]?") {
            $b64 = $matches[1]
        } else {
            return $null
        }
        
        # Attempt decode
        $decodedBytes = [System.Convert]::FromBase64String($b64)
        $decodedText = [System.Text.Encoding]::Unicode.GetString($decodedBytes)
        return $decodedText
    } catch {
        return $null
    }
}

function Test-IsSuspiciousScript {
    param([string]$ScriptContent)
    
    $suspiciousPatterns = @(
        "IEX\s*\(|Invoke-Expression",
        "DownloadString",
        "Invoke-WebRequest.*http",
        "Invoke-RestMethod.*http",
        "Net\.WebClient",
        "System\.Net\.WebClient",
        "amsi\.dll",
        "AmsiScanBuffer",
        "AmsiUtils",
        "Reflection\.Assembly",
        "Add-Type.*-MemberDefinition",
        "FromBase64String",
        "obfuscated",
        "bypass",
        "hidden",
        "-WindowStyle Hidden",
        "-ExecutionPolicy Bypass"
    )
    
    foreach ($pattern in $suspiciousPatterns) {
        if ($ScriptContent -match $pattern) {
            return $true
        }
    }
    return $false
}

# ==============================================================================
# Enhanced Detection Functions
# ==============================================================================

function Detect-EncodedPowerShell {
    Write-Log "Detect-EncodedPowerShell: Starting scan"
    
    # Check Scheduled Tasks for encoded commands
    try {
        $tasks = Get-ScheduledTask -ErrorAction SilentlyContinue | Where-Object { $_.TaskName -notlike "\Microsoft*" } | Select-Object TaskName, Actions, Author, Principal
        
        foreach ($task in $tasks) {
            if ($task.Actions) {
                foreach ($action in $task.Actions) {
                    if ($action.Execute -like "*powershell*" -and $action.Arguments -match "-EncodedCommand|-e\s") {
                        
                        # Check if allowlisted
                        if (Test-IsAllowlisted $task.TaskName) {
                            Write-Log "Detect-EncodedPowerShell: Allowlisted task $($task.TaskName)"
                            continue
                        }
                        
                        # Attempt to decode
                        $decoded = Decode-Base64IfPossible $action.Arguments
                        $isSuspicious = $false
                        $additionalEvidence = ""
                        
                        if ($decoded) {
                            $isSuspicious = Test-IsSuspiciousScript $decoded
                            $additionalEvidence = "`nDecoded: $($decoded.Substring(0, [Math]::Min(200, $decoded.Length)))..."
                        }
                        
                        $risk = if ($isSuspicious) { "High" } else { "Medium" }
                        $description = if ($isSuspicious) { 
                            "Scheduled task contains encoded PowerShell with suspicious content." 
                        } else { 
                            "Scheduled task contains encoded PowerShell command (common LOTL technique)." 
                        }
                        
                        Add-Finding -Category "PowerShell" -Title "Encoded PowerShell in Scheduled Task" `
                            -Description $description `
                            -Risk $risk `
                            -Evidence "Task: $($task.TaskName)`nAuthor: $($task.Author)`nArguments: $($action.Arguments)$additionalEvidence" `
                            -Remediation "Review scheduled task '$($task.TaskName)' for legitimacy. Check decoded content if available."
                    }
                }
            }
        }
    } catch {
        Write-Log "Detect-EncodedPowerShell: Error checking scheduled tasks: $_"
    }
    
    # Check Registry for encoded command persistence
    try {
        $regPaths = @(
            "HKLM:\Software\Microsoft\Windows\CurrentVersion\Run",
            "HKLM:\Software\Microsoft\Windows\CurrentVersion\RunOnce",
            "HKCU:\Software\Microsoft\Windows\CurrentVersion\Run",
            "HKCU:\Software\Microsoft\Windows\CurrentVersion\RunOnce"
        )
        
        foreach ($path in $regPaths) {
            if (Test-Path $path) {
                $values = Get-ItemProperty -Path $path -ErrorAction SilentlyContinue
                foreach ($value in $values.PSObject.Properties | Where-Object { $_.Name -notin @("PSPath", "PSParentPath", "PSChildName", "PSDrive", "PSProvider") }) {
                    if ($value.Value -match "powershell.*-EncodedCommand|-e\s") {
                        
                        # Attempt to decode
                        $decoded = Decode-Base64IfPossible $value.Value
                        $isSuspicious = $false
                        $additionalEvidence = ""
                        
                        if ($decoded) {
                            $isSuspicious = Test-IsSuspiciousScript $decoded
                            $additionalEvidence = "`nDecoded: $($decoded.Substring(0, [Math]::Min(200, $decoded.Length)))..."
                        }
                        
                        $risk = if ($isSuspicious) { "High" } else { "Medium" }
                        
                        Add-Finding -Category "PowerShell" -Title "Encoded PowerShell in Registry" `
                            -Description "Registry run key contains encoded PowerShell command." `
                            -Risk $risk `
                            -Evidence "Registry: $path\$($value.Name)`nValue: $($value.Value)$additionalEvidence" `
                            -Remediation "Review registry entry and remove if suspicious."
                    }
                }
            }
        }
    } catch {
        Write-Log "Detect-EncodedPowerShell: Error checking registry: $_"
    }
}

function Detect-ScheduledTasks {
    Write-Log "Detect-ScheduledTasks: Starting scan"
    
    try {
        $suspiciousTasks = Get-ScheduledTask -ErrorAction SilentlyContinue | Where-Object {
            $_.TaskName -notlike "\Microsoft*" -and
            (Test-IsAllowlisted $_.TaskName -eq $false)
        }
        
        foreach ($task in $suspiciousTasks) {
            foreach ($action in $task.Actions) {
                $isSuspicious = $false
                $suspiciousReason = ""
                
                # Check for execution from temp locations
                if ($action.Execute -match "(?i)\\temp\\|\\tmp\\|%temp%|%tmp%") {
                    $isSuspicious = $true
                    $suspiciousReason = "Executes from temporary location"
                }
                
                # Check for script files
                if ($action.Execute -match "\.(vbs|js|ps1|hta|bat|cmd)$") {
                    $isSuspicious = $true
                    $suspiciousReason = "Executes script file"
                }
                
                # Check for unsigned executables in suspicious locations
                if ($action.Execute -match "\.exe$") {
                    $filePath = $action.Execute
                    if (Test-Path $filePath) {
                        try {
                            $signature = Get-AuthenticodeSignature -FilePath $filePath -ErrorAction SilentlyContinue
                            if ($signature.Status -ne "Valid") {
                                $isSuspicious = $true
                                $suspiciousReason = "Unsigned executable"
                            }
                        } catch {
                            # Can't check signature
                        }
                    }
                }
                
                if ($isSuspicious) {
                    Add-Finding -Category "ScheduledTask" -Title "Suspicious Scheduled Task" `
                        -Description "Scheduled task has suspicious characteristics: $suspiciousReason." `
                        -Risk "Medium" `
                        -Evidence "Task: $($task.TaskName)`nAuthor: $($task.Author)`nExecute: $($action.Execute)`nArguments: $($action.Arguments)" `
                        -Remediation "Investigate scheduled task '$($task.TaskName)' for legitimacy. Check file location and signature."
                }
            }
        }
    } catch {
        Write-Log "Detect-ScheduledTasks: Error analyzing scheduled tasks: $_"
    }
}

function Detect-WMIPersistence {
    Write-Log "Detect-WMIPersistence: Starting scan"
    
    try {
        # Check for all event consumer types
        $consumerClasses = @("ActiveScriptEventConsumer", "CommandLineEventConsumer", "LogFileEventConsumer", "NTEventLogEventConsumer")
        
        foreach ($className in $consumerClasses) {
            $consumers = Get-WmiObject -Namespace "root\subscription" -Class $className -ErrorAction SilentlyContinue
            
            foreach ($consumer in $consumers) {
                $isSuspicious = $false
                
                if ($className -eq "ActiveScriptEventConsumer") {
                    if ($consumer.ScriptText -match "(?i)IEX|DownloadString|Invoke-") {
                        $isSuspicious = $true
                    }
                } elseif ($className -eq "CommandLineEventConsumer") {
                    if ($consumer.CommandLineTemplate -match "(?i)powershell|cmd\.exe.*/c") {
                        $isSuspicious = $true
                    }
                }
                
                if ($isSuspicious) {
                    Add-Finding -Category "WMI" -Title "Suspicious WMI Event Consumer" `
                        -Description "WMI $className detected with potentially malicious content." `
                        -Risk "High" `
                        -Evidence "Consumer: $($consumer.Name)`nClass: $className`nDetails: $(if ($consumer.ScriptText) { $consumer.ScriptText } elseif ($consumer.CommandLineTemplate) { $consumer.CommandLineTemplate } else { 'N/A' })" `
                        -Remediation "Review WMI event subscriptions in root\subscription namespace. Remove suspicious consumers."
                }
            }
        }
        
        # Check for filter-to-consumer bindings (permanent event subscriptions)
        $bindings = Get-WmiObject -Namespace "root\subscription" -Class "__FilterToConsumerBinding" -ErrorAction SilentlyContinue
        
        if ($bindings) {
            foreach ($binding in $bindings) {
                $filter = Get-WmiObject -Namespace "root\subscription" -Query "SELECT * FROM __EventFilter WHERE Name='$($binding.Filter)'" -ErrorAction SilentlyContinue
                $consumer = Get-WmiObject -Namespace "root\subscription" -Query "SELECT * FROM __EventConsumer WHERE Name='$($binding.Consumer)'" -ErrorAction SilentlyContinue
                
                if ($filter -and $consumer) {
                    Add-Finding -Category "WMI" -Title "WMI Permanent Event Subscription" `
                        -Description "WMI permanent event subscription detected (common persistence mechanism)." `
                        -Risk "High" `
                        -Evidence "Filter: $($filter.Name) ($($filter.Query))`nConsumer: $($consumer.Name) ($($consumer.__CLASS))" `
                        -Remediation "Review and remove suspicious WMI event subscriptions: `nGet-WmiObject -Namespace root\subscription -Class __FilterToConsumerBinding | Remove-WmiObject"
                }
            }
        }
    } catch {
        Write-Log "Detect-WMIPersistence: Error checking WMI: $_"
    }
}

function Detect-RegistryAnomalies {
    Write-Log "Detect-RegistryAnomalies: Starting scan"
    
    try {
        $suspiciousPatterns = @(
            "\.(vbs|js|ps1|hta|bat|cmd)$",
            "(?i)\\temp\\",
            "(?i)\\tmp\\",
            "rundll32.*\.dll.*,.*",
            "regsvr32.*/s.*\.(dll|ocx)",
            "mshta.*\.(hta|url|http)",
            "wscript.*\.(vbs|js)",
            "cscript.*\.(vbs|js)"
        )
        
        $regPaths = @(
            "HKLM:\Software\Microsoft\Windows\CurrentVersion\Run",
            "HKLM:\Software\Microsoft\Windows\CurrentVersion\RunOnce",
            "HKCU:\Software\Microsoft\Windows\CurrentVersion\Run",
            "HKCU:\Software\Microsoft\Windows\CurrentVersion\RunOnce"
        )
        
        foreach ($path in $regPaths) {
            if (Test-Path $path) {
                $values = Get-ItemProperty -Path $path -ErrorAction SilentlyContinue
                foreach ($value in $values.PSObject.Properties | Where-Object { $_.Name -notin @("PSPath", "PSParentPath", "PSChildName", "PSDrive", "PSProvider") }) {
                    
                    # Check if allowlisted
                    if (Test-IsAllowlisted $value.Name) {
                        Write-Log "Detect-RegistryAnomalies: Allowlisted registry entry $($value.Name)"
                        continue
                    }
                    
                    foreach ($pattern in $suspiciousPatterns) {
                        if ($value.Value -match $pattern) {
                            
                            # Additional checks for file existence and signature
                            $fileCheck = ""
                            if ($value.Value -match '"[^"]+"|^\S+') {
                                $exePath = if ($value.Value -match '"([^"]+)"') { $matches[1] } else { $value.Value.Split(' ')[0] }
                                if (Test-Path $exePath) {
                                    try {
                                        $signature = Get-AuthenticodeSignature -FilePath $exePath -ErrorAction SilentlyContinue
                                        if ($signature.Status -ne "Valid") {
                                            $fileCheck = "`nFile exists but unsigned: $exePath"
                                        } else {
                                            $fileCheck = "`nFile signed: $exePath"
                                        }
                                    } catch {
                                        $fileCheck = "`nFile exists: $exePath"
                                    }
                                } else {
                                    $fileCheck = "`nFile not found: $exePath"
                                }
                            }
                            
                            Add-Finding -Category "Registry" -Title "Suspicious Registry Run Entry" `
                                -Description "Registry run key contains potentially suspicious executable/script." `
                                -Risk "Medium" `
                                -Evidence "Registry: $path\$($value.Name)`nValue: $($value.Value)$fileCheck" `
                                -Remediation "Review registry entry and investigate file. Check for digital signature."
                            break
                        }
                    }
                }
            }
        }
    } catch {
        Write-Log "Detect-RegistryAnomalies: Error scanning registry run keys: $_"
    }
}

function Detect-LOLBASUsage {
    Write-Log "Detect-LOLBASUsage: Starting scan"
    
    try {
        $psHistoryPath = "$env:APPDATA\Microsoft\Windows\PowerShell\PSReadLine\ConsoleHost_history.txt"
        
        if (Test-Path $psHistoryPath) {
            $psHistory = Get-Content $psHistoryPath -ErrorAction SilentlyContinue | Select-Object -Last 100
        } else {
            $psHistory = @()
            Add-Finding -Category "LOLBAS" -Title "PowerShell History Not Found" `
                -Description "PowerShell history file not found (may be disabled or cleared)." `
                -Risk "Low" `
                -Evidence "File not found: $psHistoryPath" `
                -Remediation "Consider enabling PowerShell Script Block Logging (Event ID 4104) for better detection."
        }
        
        $lolbasPatterns = @(
            "certutil.*-decode",
            "certutil.*-urlcache",
            "bitsadmin.*/transfer",
            "regsvr32.*/s.*\.sct",
            "mshta.*http",
            "rundll32.*\.dll.*,.*",
            "wmic.*process.*call.*create",
            "wscript.*\.(vbs|js)",
            "cscript.*\.(vbs|js)",
            "SyncAppvPublishingServer\.vbs",
            "fodhelper\.exe",
            "schtasks.*/create",
            "sc.*create"
        )
        
        foreach ($line in $psHistory) {
            foreach ($pattern in $lolbasPatterns) {
                if ($line -match $pattern) {
                    Add-Finding -Category "LOLBAS" -Title "Potential LOLBAS Usage in PowerShell History" `
                        -Description "PowerShell history contains command matching known LOLBAS pattern." `
                        -Risk "Medium" `
                        -Evidence "Command: $line" `
                        -Remediation "Review PowerShell history and investigate command context. Consider enabling PowerShell Script Block Logging."
                    break
                }
            }
        }
        
        # Check for BITS jobs
        try {
            $bitsJobs = Get-BitsTransfer -ErrorAction SilentlyContinue
            if ($bitsJobs) {
                foreach ($job in $bitsJobs) {
                    if ($job.DisplayName -match "(?i)update|upgrade|install") {
                        # Possibly legitimate
                    } else {
                        Add-Finding -Category "LOLBAS" -Title "BITS Job Detected" `
                            -Description "Background Intelligent Transfer Service (BITS) job found (can be used for stealthy downloads)." `
                            -Risk "Low" `
                            -Evidence "BITS Job: $($job.DisplayName)`nState: $($job.JobState)" `
                            -Remediation "Review BITS jobs: Get-BitsTransfer | Remove-BitsTransfer if suspicious."
                    }
                }
            }
        } catch {
            # BITS module not available
        }
        
    } catch {
        Write-Log "Detect-LOLBASUsage: Error checking LOLBAS indicators: $_"
    }
}

function Detect-ProcessAncestry {
    Write-Log "Detect-ProcessAncestry: Starting scan"
    
    try {
        # Use Win32_Process for better ancestry chains
        $processes = Get-CimInstance -ClassName Win32_Process -ErrorAction SilentlyContinue | 
            Select-Object Name, ProcessId, ParentProcessId, CommandLine
        
        $processMap = @{}
        foreach ($proc in $processes) {
            $processMap[$proc.ProcessId] = $proc
        }
        
        $suspiciousParents = @("svchost.exe", "services.exe", "explorer.exe", "rundll32.exe", "regsvr32.exe", "mshta.exe", "wscript.exe", "cscript.exe")
        
        foreach ($proc in $processes) {
            if ($proc.Name -in @("powershell.exe", "cmd.exe", "wmic.exe", "bitsadmin.exe", "certutil.exe")) {
                if ($processMap.ContainsKey($proc.ParentProcessId)) {
                    $parent = $processMap[$proc.ParentProcessId]
                    
                    if ($parent.Name -in $suspiciousParents) {
                        $risk = if ($parent.Name -in @("rundll32.exe", "regsvr32.exe", "mshta.exe")) { "Medium" } else { "Low" }
                        
                        Add-Finding -Category "Process" -Title "Suspicious Process Ancestry" `
                            -Description "$($proc.Name) spawned by $($parent.Name) (unusual parent-child relationship)." `
                            -Risk $risk `
                            -Evidence "Process: $($proc.Name) (PID: $($proc.ProcessId))`nParent: $($parent.Name) (PID: $($proc.ParentProcessId))`nCommand Line: $($proc.CommandLine)" `
                            -Remediation "Investigate process and parent relationship. Check if this is expected behavior."
                    }
                }
            }
        }
    } catch {
        Write-Log "Detect-ProcessAncestry: Error analyzing process ancestry: $_"
    }
}

function Detect-PowerShellLogging {
    Write-Log "Detect-PowerShellLogging: Starting scan"
    
    # Check if PowerShell logging is enabled
    try {
        $scriptBlockLogging = Get-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ScriptBlockLogging" -Name "EnableScriptBlockLogging" -ErrorAction SilentlyContinue
        $moduleLogging = Get-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ModuleLogging" -Name "EnableModuleLogging" -ErrorAction SilentlyContinue
        
        if (-not $scriptBlockLogging -or $scriptBlockLogging.EnableScriptBlockLogging -ne 1) {
            Add-Finding -Category "Configuration" -Title "PowerShell Script Block Logging Not Enabled" `
                -Description "PowerShell Script Block Logging (Event ID 4104) is not enabled, limiting detection capabilities." `
                -Risk "Low" `
                -Evidence "Registry key HKLM:\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ScriptBlockLogging\EnableScriptBlockLogging not set to 1" `
                -Remediation "Enable PowerShell Script Block Logging for better LOTL detection: `nSet-ItemProperty -Path 'HKLM:\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ScriptBlockLogging' -Name 'EnableScriptBlockLogging' -Value 1"
        }
        
        if (-not $moduleLogging -or $moduleLogging.EnableModuleLogging -ne 1) {
            Add-Finding -Category "Configuration" -Title "PowerShell Module Logging Not Enabled" `
                -Description "PowerShell Module Logging is not enabled, limiting visibility into PowerShell activity." `
                -Risk "Low" `
                -Evidence "Registry key HKLM:\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ModuleLogging\EnableModuleLogging not set to 1" `
                -Remediation "Enable PowerShell Module Logging: `nSet-ItemProperty -Path 'HKLM:\SOFTWARE\Policies\Microsoft\Windows\PowerShell\ModuleLogging' -Name 'EnableModuleLogging' -Value 1"
        }
    } catch {
        Write-Log "Detect-PowerShellLogging: Error checking PowerShell logging configuration: $_"
    }
}

function Invoke-FullDetection {
    param(
        [ValidateSet("Basic", "Medium", "Full")]
        [string]$ScanDepth = "Basic"
    )
    
    Clear-Findings
    
    Write-Log "Invoke-FullDetection: Starting scan with depth $ScanDepth"
    
    # Check if running as admin (recommended)
    if (-not (Test-IsAdmin)) {
        Add-Finding -Category "Configuration" -Title "Not Running as Administrator" `
            -Description "LOT-Squatch is not running with administrator privileges, which may limit detection capabilities." `
            -Risk "Low" `
            -Evidence "Current user does not have administrator rights." `
            -Remediation "Run LOT-Squatch as Administrator for full detection capabilities."
    }
    
    Detect-EncodedPowerShell
    Detect-ScheduledTasks
    Detect-WMIPersistence
    Detect-RegistryAnomalies
    Detect-LOLBASUsage
    
    if ($ScanDepth -in @("Medium", "Full")) {
        Detect-ProcessAncestry
    }
    
    if ($ScanDepth -eq "Full") {
        Detect-PowerShellLogging
    }
    
    Write-Log "Invoke-FullDetection: Scan complete, found $($Script:Findings.Count) findings"
    return Get-Findings
}

function Write-Log {
    param([string]$Message)
    
    # Log to file if log path is set, otherwise do nothing
    if ($Script:LogPath) {
        $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
        "$timestamp | $Message" | Out-File -FilePath $Script:LogPath -Append -Encoding UTF8
    }
}

# Initialize log path (can be overridden by calling script)
$Script:LogPath = "$env:TEMP\LOT-Squatch-Detection-$(Get-Date -Format 'yyyyMMdd-HHmmss').log"